// Copyright 2020-2023 Mesiontech Technology Co., Ltd. All Rights Reserved.

#include "YyssXRInput.h"
#include "YyssXRInputState.h"
#include "YyssXRHMD.h"
#include "YyssXRCore.h"
#include "UObject/UObjectIterator.h"
#include "GameFramework/InputSettings.h"
#include "HeadMountedDisplayFunctionLibrary.h"

#if WITH_EDITOR
#include "Editor/EditorEngine.h"
#include "Editor.h"
#endif

#include "openxr.h"

#define LOCTEXT_NAMESPACE "YyssXRInputPlugin"

using namespace YyssInput;


//**************	Key	***************************
const FKey FYyssKey::Yyss_Left_X_Click("Yyss_Left_X_Click");
const FKey FYyssKey::Yyss_Left_Y_Click("Yyss_Left_Y_Click");
const FKey FYyssKey::Yyss_Left_Back_Click("Yyss_Left_Back_Click");
const FKey FYyssKey::Yyss_Left_Home_Click("Yyss_Left_Home_Click");
const FKey FYyssKey::Yyss_Left_Trigger_Click("Yyss_Left_Trigger_Click");
const FKey FYyssKey::Yyss_Left_Grip_Click("Yyss_Left_Grip_Click");
const FKey FYyssKey::Yyss_Left_Rocker_Click("Yyss_Left_Rocker_Click");
const FKey FYyssKey::Yyss_Left_Rocker_Up("Yyss_Left_Rocker_Up");
const FKey FYyssKey::Yyss_Left_Rocker_Down("Yyss_Left_Rocker_Down");
const FKey FYyssKey::Yyss_Left_Rocker_Left("Yyss_Left_Rocker_Left");
const FKey FYyssKey::Yyss_Left_Rocker_Right("Yyss_Left_Rocker_Right");

const FKey FYyssKey::Yyss_Right_A_Click("Yyss_Right_A_Click");
const FKey FYyssKey::Yyss_Right_B_Click("Yyss_Right_B_Click");
const FKey FYyssKey::Yyss_Right_Back_Click("Yyss_Right_Back_Click");
const FKey FYyssKey::Yyss_Right_Home_Click("Yyss_Right_Home_Click");
const FKey FYyssKey::Yyss_Right_Trigger_Click("Yyss_Right_Trigger_Click");
const FKey FYyssKey::Yyss_Right_Grip_Click("Yyss_Right_Grip_Click");
const FKey FYyssKey::Yyss_Right_Rocker_Click("Yyss_Right_Rocker_Click");
const FKey FYyssKey::Yyss_Right_Rocker_Up("Yyss_Right_Rocker_Up");
const FKey FYyssKey::Yyss_Right_Rocker_Down("Yyss_Right_Rocker_Down");
const FKey FYyssKey::Yyss_Right_Rocker_Left("Yyss_Right_Rocker_Left");
const FKey FYyssKey::Yyss_Right_Rocker_Right("Yyss_Right_Rocker_Right");

const FKey FYyssKey::Yyss_Left_X_Touch("Yyss_Left_X_Touch");
const FKey FYyssKey::Yyss_Left_Y_Touch("Yyss_Left_Y_Touch");
const FKey FYyssKey::Yyss_Left_Trigger_Touch("Yyss_Left_Trigger_Touch");
const FKey FYyssKey::Yyss_Left_Rocker_Touch("Yyss_Left_Rocker_Touch");

const FKey FYyssKey::Yyss_Right_A_Touch("Yyss_Right_A_Touch");
const FKey FYyssKey::Yyss_Right_B_Touch("Yyss_Right_B_Touch");
const FKey FYyssKey::Yyss_Right_Trigger_Touch("Yyss_Right_Trigger_Touch");
const FKey FYyssKey::Yyss_Right_Rocker_Touch("Yyss_Right_Rocker_Touch");

const FKey FYyssKey::Yyss_Left_Trigger_Axis("Yyss_Left_Trigger_Axis");
const FKey FYyssKey::Yyss_Left_Grip_Axis("Yyss_Left_Grip_Axis");
const FKey FYyssKey::Yyss_Left_Rocker_X("Yyss_Left_Rocker_X");
const FKey FYyssKey::Yyss_Left_Rocker_Y("Yyss_Left_Rocker_Y");

const FKey FYyssKey::Yyss_Right_Trigger_Axis("Yyss_Right_Trigger_Axis");
const FKey FYyssKey::Yyss_Right_Grip_Axis("Yyss_Right_Grip_Axis");
const FKey FYyssKey::Yyss_Right_Rocker_X("Yyss_Right_Rocker_X");
const FKey FYyssKey::Yyss_Right_Rocker_Y("Yyss_Right_Rocker_Y");

//**************	Name	***************************
const FName FYyssKeyNames::Yyss_Left_X_Click("Yyss_Left_X_Click");
const FName FYyssKeyNames::Yyss_Left_Y_Click("Yyss_Left_Y_Click");
const FName FYyssKeyNames::Yyss_Left_Back_Click("Yyss_Left_Back_Click");
const FName FYyssKeyNames::Yyss_Left_Home_Click("Yyss_Left_Home_Click");
const FName FYyssKeyNames::Yyss_Left_Trigger_Click("Yyss_Left_Trigger_Click");
const FName FYyssKeyNames::Yyss_Left_Grip_Click("Yyss_Left_Grip_Click");
const FName FYyssKeyNames::Yyss_Left_Rocker_Click("Yyss_Left_Rocker_Click");
const FName FYyssKeyNames::Yyss_Left_Rocker_Up("Yyss_Left_Rocker_Up");
const FName FYyssKeyNames::Yyss_Left_Rocker_Down("Yyss_Left_Rocker_Down");
const FName FYyssKeyNames::Yyss_Left_Rocker_Left("Yyss_Left_Rocker_Left");
const FName FYyssKeyNames::Yyss_Left_Rocker_Right("Yyss_Left_Rocker_Right");

const FName FYyssKeyNames::Yyss_Right_A_Click("Yyss_Right_A_Click");
const FName FYyssKeyNames::Yyss_Right_B_Click("Yyss_Right_B_Click");
const FName FYyssKeyNames::Yyss_Right_Back_Click("Yyss_Right_Back_Click");
const FName FYyssKeyNames::Yyss_Right_Home_Click("Yyss_Right_Home_Click");
const FName FYyssKeyNames::Yyss_Right_Grip_Click("Yyss_Right_Grip_Click");
const FName FYyssKeyNames::Yyss_Right_Trigger_Click("Yyss_Right_Trigger_Click");
const FName FYyssKeyNames::Yyss_Right_Rocker_Click("Yyss_Right_Rocker_Click");
const FName FYyssKeyNames::Yyss_Right_Rocker_Up("Yyss_Right_Rocker_Up");
const FName FYyssKeyNames::Yyss_Right_Rocker_Down("Yyss_Right_Rocker_Down");
const FName FYyssKeyNames::Yyss_Right_Rocker_Left("Yyss_Right_Rocker_Left");
const FName FYyssKeyNames::Yyss_Right_Rocker_Right("Yyss_Right_Rocker_Right");

const FName FYyssKeyNames::Yyss_Left_X_Touch("Yyss_Left_X_Touch");
const FName FYyssKeyNames::Yyss_Left_Y_Touch("Yyss_Left_Y_Touch");
const FName FYyssKeyNames::Yyss_Left_Trigger_Touch("Yyss_Left_Trigger_Touch");
const FName FYyssKeyNames::Yyss_Left_Rocker_Touch("Yyss_Left_Rocker_Touch");

const FName FYyssKeyNames::Yyss_Right_A_Touch("Yyss_Right_A_Touch");
const FName FYyssKeyNames::Yyss_Right_B_Touch("Yyss_Right_B_Touch");
const FName FYyssKeyNames::Yyss_Right_Trigger_Touch("Yyss_Right_Trigger_Touch");
const FName FYyssKeyNames::Yyss_Right_Rocker_Touch("Yyss_Right_Rocker_Touch");

const FName FYyssKeyNames::Yyss_Left_Trigger_Axis("Yyss_Left_Trigger_Axis");
const FName FYyssKeyNames::Yyss_Left_Grip_Axis("Yyss_Left_Grip_Axis");
const FName FYyssKeyNames::Yyss_Left_Rocker_X("Yyss_Left_Rocker_X");
const FName FYyssKeyNames::Yyss_Left_Rocker_Y("Yyss_Left_Rocker_Y");

const FName FYyssKeyNames::Yyss_Right_Trigger_Axis("Yyss_Right_Trigger_Axis");
const FName FYyssKeyNames::Yyss_Right_Grip_Axis("Yyss_Right_Grip_Axis");
const FName FYyssKeyNames::Yyss_Right_Rocker_X("Yyss_Right_Rocker_X");
const FName FYyssKeyNames::Yyss_Right_Rocker_Y("Yyss_Right_Rocker_Y");

namespace OpenXRSourceNames
{
	static const FName AnyHand("AnyHand");
	static const FName Left("Left");
	static const FName Right("Right");
	static const FName LeftGrip("LeftGrip");
	static const FName RightGrip("RightGrip");
	static const FName LeftAim("LeftAim");
	static const FName RightAim("RightAim");
}

FORCEINLINE XrPath GetPath(XrInstance Instance, const char* PathString)
{
	XrPath Path = XR_NULL_PATH;
	XrResult Result = xrStringToPath(Instance, PathString, &Path);
	//UE_LOG(LogInput, Log, TEXT("pathsting = %s"),UTF8_TO_TCHAR(PathString));
	check(XR_SUCCEEDED(Result));
	return Path;
}

FORCEINLINE XrPath GetPath(XrInstance Instance, const FString& PathString)
{
	return GetPath(Instance, (ANSICHAR*)StringCast<ANSICHAR>(*PathString).Get());
}

FORCEINLINE void FilterActionName(const char* InActionName, char* OutActionName)
{
	// Ensure the action name is a well-formed path
	size_t i;
	for (i = 0; i < XR_MAX_ACTION_NAME_SIZE - 1 && InActionName[i] != '\0'; i++)
	{
		unsigned char c = InActionName[i];
		OutActionName[i] = (c == ' ') ? '-' : isalnum(c) ? tolower(c) : '_';
	}
	OutActionName[i] = '\0';
}

TSharedPtr< class IInputDevice > FYyssXRInputPlugin::CreateInputDevice(const TSharedRef< FGenericApplicationMessageHandler >& InMessageHandler)
{
	if (InputDevice)
		InputDevice->SetMessageHandler(InMessageHandler);
	return InputDevice;
}

IMPLEMENT_MODULE(FYyssXRInputPlugin, YyssXRInput)

FYyssXRInputPlugin::FYyssXRInputPlugin()
	: InputDevice()
{
}

FYyssXRInputPlugin::~FYyssXRInputPlugin()
{
}

FYyssXRHMD* FYyssXRInputPlugin::GetOpenXRHMD() const
{
	static FName SystemName(TEXT("YyssXR"));//todo
	if (GEngine->XRSystem.IsValid() && (GEngine->XRSystem->GetSystemName() == SystemName)) 
	{
		return static_cast<FYyssXRHMD*>(GEngine->XRSystem.Get());
	}

	return nullptr;
}

void FYyssXRInputPlugin::StartupModule()
{
	IYyssXRInputPlugin::StartupModule();

	FYyssXRHMD* OpenXRHMD = GetOpenXRHMD();
	// Note: OpenXRHMD may be null, for example in the editor.  But we still need the input device to enumerate sources.
	InputDevice = MakeShared<FYyssXRInput>(OpenXRHMD);
}

FYyssXRInputPlugin::FOpenXRAction::FOpenXRAction(XrActionSet InActionSet, XrActionType InActionType, const FName& InName)
	: Set(InActionSet)
	, Type(InActionType)
	, Name(InName)
	, Handle(XR_NULL_HANDLE)
	, bIsTriggerButton(false)
	, bTriggerButtonPressed(false)
{
	char ActionName[NAME_SIZE];
	Name.GetPlainANSIString(ActionName);

	XrActionCreateInfo Info;
	Info.type = XR_TYPE_ACTION_CREATE_INFO;
	Info.next = nullptr;
	FilterActionName(ActionName, Info.actionName);
	Info.actionType = Type;
	Info.countSubactionPaths = 0;
	Info.subactionPaths = nullptr;
	FCStringAnsi::Strcpy(Info.localizedActionName, XR_MAX_LOCALIZED_ACTION_NAME_SIZE, ActionName);
	XR_ENSURE(xrCreateAction(Set, &Info, &Handle));
}

FYyssXRInputPlugin::FOpenXRController::FOpenXRController(XrActionSet InActionSet, XrPath InUserPath, const char* InName)
	: ActionSet(InActionSet)
	, UserPath(InUserPath)
	, GripAction(XR_NULL_HANDLE)
	, AimAction(XR_NULL_HANDLE)
	, VibrationAction(XR_NULL_HANDLE)
	, GripDeviceId(-1)
	, AimDeviceId(-1)
{
	XrActionCreateInfo Info;
	Info.type = XR_TYPE_ACTION_CREATE_INFO;
	Info.next = nullptr;
	Info.countSubactionPaths = 0;
	Info.subactionPaths = nullptr;

	FCStringAnsi::Strcpy(Info.localizedActionName, XR_MAX_ACTION_NAME_SIZE, InName);
	FCStringAnsi::Strcat(Info.localizedActionName, XR_MAX_ACTION_NAME_SIZE, " Grip Pose");
	FilterActionName(Info.localizedActionName, Info.actionName);
	Info.actionType = XR_ACTION_TYPE_POSE_INPUT;
	XR_ENSURE(xrCreateAction(ActionSet, &Info, &GripAction));

	FCStringAnsi::Strcpy(Info.localizedActionName, XR_MAX_ACTION_NAME_SIZE, InName);
	FCStringAnsi::Strcat(Info.localizedActionName, XR_MAX_ACTION_NAME_SIZE, " Aim Pose");
	FilterActionName(Info.localizedActionName, Info.actionName);
	Info.actionType = XR_ACTION_TYPE_POSE_INPUT;
	XR_ENSURE(xrCreateAction(ActionSet, &Info, &AimAction));

	FCStringAnsi::Strcpy(Info.localizedActionName, XR_MAX_ACTION_NAME_SIZE, InName);
	FCStringAnsi::Strcat(Info.localizedActionName, XR_MAX_ACTION_NAME_SIZE, " Vibration");
	FilterActionName(Info.localizedActionName, Info.actionName);
	Info.actionType = XR_ACTION_TYPE_VIBRATION_OUTPUT;
	XR_ENSURE(xrCreateAction(ActionSet, &Info, &VibrationAction));
}

void FYyssXRInputPlugin::FOpenXRController::AddActionDevices(FYyssXRHMD* HMD)
{
	if (HMD)
	{
		GripDeviceId = HMD->AddActionDevice(GripAction, UserPath);
		AimDeviceId = HMD->AddActionDevice(AimAction, UserPath);
	}
}

FYyssXRInputPlugin::FInteractionProfile::FInteractionProfile(XrPath InProfile, bool InHasHaptics)
	: HasHaptics(InHasHaptics)
	, Path(InProfile)
	, Bindings()
{
}

FYyssXRInputPlugin::FYyssXRInput::FYyssXRInput(FYyssXRHMD* HMD)
	: OpenXRHMD(HMD)
	, ActionSets()
	, Actions()
	, Controllers()
	, bActionsBound(false)
	, MessageHandler(new FGenericApplicationMessageHandler())
{
	AddKeysToEngine();
	IModularFeatures::Get().RegisterModularFeature(GetModularFeatureName(), this);

	// If there is no HMD then this module is not active, but it still needs to exist so we can EnumerateMotionSources from it.
	if (OpenXRHMD)
	{
		// Note: AnyHand needs special handling because it tries left then falls back to right in each call.
		MotionSourceToControllerHandMap.Add(OpenXRSourceNames::Left, EControllerHand::Left);
		MotionSourceToControllerHandMap.Add(OpenXRSourceNames::Right, EControllerHand::Right);
		MotionSourceToControllerHandMap.Add(OpenXRSourceNames::LeftGrip, EControllerHand::Left);
		MotionSourceToControllerHandMap.Add(OpenXRSourceNames::RightGrip, EControllerHand::Right);
		MotionSourceToControllerHandMap.Add(OpenXRSourceNames::LeftAim, EControllerHand::Left);
		MotionSourceToControllerHandMap.Add(OpenXRSourceNames::RightAim, EControllerHand::Right);

		// Map the legacy hand enum values that openxr supports
		MotionSourceToControllerHandMap.Add(TEXT("EControllerHand::Left"), EControllerHand::Left);
		MotionSourceToControllerHandMap.Add(TEXT("EControllerHand::Right"), EControllerHand::Right);
		MotionSourceToControllerHandMap.Add(TEXT("EControllerHand::AnyHand"), EControllerHand::AnyHand);

		BuildActions();
	}
}

XrAction FYyssXRInputPlugin::FYyssXRInput::GetActionForMotionSource(FName MotionSource) const
{
	const FOpenXRController& Controller = Controllers[MotionSourceToControllerHandMap.FindChecked(MotionSource)];
	if (MotionSource == OpenXRSourceNames::LeftAim || MotionSource == OpenXRSourceNames::RightAim)
	{
		return Controller.AimAction;
	}
	else
	{
		return Controller.GripAction;
	}
}

int32 FYyssXRInputPlugin::FYyssXRInput::GetDeviceIDForMotionSource(FName MotionSource) const
{
	const FOpenXRController& Controller = Controllers[MotionSourceToControllerHandMap.FindChecked(MotionSource)];
	if (MotionSource == OpenXRSourceNames::LeftAim || MotionSource == OpenXRSourceNames::RightAim)
	{
		return Controller.AimDeviceId;
	}
	else
	{
		return Controller.GripDeviceId;
	}
}

XrPath FYyssXRInputPlugin::FYyssXRInput::GetUserPathForMotionSource(FName MotionSource) const
{
	const FOpenXRController& Controller = Controllers[MotionSourceToControllerHandMap.FindChecked(MotionSource)];
	return Controller.UserPath;
}

bool FYyssXRInputPlugin::FYyssXRInput::IsOpenXRInputSupportedMotionSource(const FName MotionSource) const
{
	return MotionSource == OpenXRSourceNames::AnyHand || MotionSourceToControllerHandMap.Contains(MotionSource);
}

FYyssXRInputPlugin::FYyssXRInput::~FYyssXRInput()
{
	DestroyActions();
}

void FYyssXRInputPlugin::FYyssXRInput::BuildActions()
{
	if ((bActionsBound) || (OpenXRHMD == nullptr))
	{
		return;
	}
	
	XrInstance Instance = OpenXRHMD->GetInstance();
	check(Instance);
	DestroyActions();

	XrActionSet ActionSet = XR_NULL_HANDLE;
	XrActionSetCreateInfo SetInfo;
	SetInfo.type = XR_TYPE_ACTION_SET_CREATE_INFO;
	SetInfo.next = nullptr;
	FCStringAnsi::Strcpy(SetInfo.actionSetName, XR_MAX_ACTION_SET_NAME_SIZE, "ue4");
	FCStringAnsi::Strcpy(SetInfo.localizedActionSetName, XR_MAX_ACTION_SET_NAME_SIZE, "Unreal Engine 4");
	SetInfo.priority = 0;
	XR_ENSURE(xrCreateActionSet(Instance, &SetInfo, &ActionSet));

	XrPath LeftHand = GetPath(Instance, "/user/hand/left");
	XrPath RightHand = GetPath(Instance, "/user/hand/right");

	// Controller poses
	OpenXRHMD->ResetActionDevices();
	Controllers.Add(EControllerHand::Left, FOpenXRController(ActionSet, LeftHand, "Left Controller"));
	Controllers.Add(EControllerHand::Right, FOpenXRController(ActionSet, RightHand, "Right Controller"));

	// Generate a map of all supported interaction profiles
	XrPath SimpleControllerPath = GetPath(Instance, "/interaction_profiles/khr/simple_controller");
	Profiles.Add("SimpleController", FInteractionProfile(SimpleControllerPath, true));
	
	Profiles.Add("Yyss", FInteractionProfile(GetPath(Instance, "/interaction_profiles/oculus/touch_controller"), false));
	//left click
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Left_X_Click, "/user/hand/left/input/x/click");
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Left_Y_Click, "/user/hand/left/input/y/click");
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Left_Back_Click, "/user/hand/left/input/back/click");	
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Left_Home_Click, "/user/hand/left/input/home/click");
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Left_Grip_Click, "/user/hand/left/input/squeeze/value");
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Left_Rocker_Click, "/user/hand/left/input/thumbstick/click");
	//right click
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Right_A_Click, "/user/hand/right/input/a/click");
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Right_B_Click, "/user/hand/right/input/b/click");
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Right_Back_Click, "/user/hand/right/input/back/click");	
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Right_Home_Click, "/user/hand/right/input/home/click");
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Right_Grip_Click, "/user/hand/right/input/squeeze/value");
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Right_Rocker_Click, "/user/hand/right/input/thumbstick/click");
	//left touch
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Left_Trigger_Touch, "/user/hand/left/input/trigger/touch");
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Left_X_Touch, "/user/hand/left/input/x/touch");
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Left_Y_Touch, "/user/hand/left/input/y/touch");
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Left_Rocker_Touch, "/user/hand/left/input/thumbstick/touch");
	//right touch
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Right_Trigger_Touch, "/user/hand/right/input/trigger/touch");
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Right_A_Touch, "/user/hand/right/input/a/touch");
	// AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Right_B_Touch, "/user/hand/right/input/b/touch");
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_BOOLEAN_INPUT, FYyssKeyNames::Yyss_Right_Rocker_Touch, "/user/hand/right/input/thumbstick/touch");
	//left axis
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_FLOAT_INPUT, FYyssKeyNames::Yyss_Left_Trigger_Click, "/user/hand/left/input/trigger/value", true);
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_FLOAT_INPUT, FYyssKeyNames::Yyss_Left_Trigger_Axis, "/user/hand/left/input/trigger/value");
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_FLOAT_INPUT, FYyssKeyNames::Yyss_Left_Rocker_X, "/user/hand/left/input/thumbstick/x");
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_FLOAT_INPUT, FYyssKeyNames::Yyss_Left_Rocker_Y, "/user/hand/left/input/thumbstick/y"); 
	//right axis
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_FLOAT_INPUT, FYyssKeyNames::Yyss_Right_Trigger_Click, "/user/hand/right/input/trigger/value", true);
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_FLOAT_INPUT, FYyssKeyNames::Yyss_Right_Trigger_Axis, "/user/hand/right/input/trigger/value");
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_FLOAT_INPUT, FYyssKeyNames::Yyss_Right_Rocker_X, "/user/hand/right/input/thumbstick/x");
	AddYyssAction(Instance, ActionSet, XR_ACTION_TYPE_FLOAT_INPUT, FYyssKeyNames::Yyss_Right_Rocker_Y, "/user/hand/right/input/thumbstick/y");

	for (TPair<FString, FInteractionProfile>& Pair : Profiles)
	{
		FInteractionProfile& Profile = Pair.Value;
		// Only suggest interaction profile bindings if the developer has provided bindings for them
		// An exception is made for the Simple Controller Profile which is always bound as a fallback
		if (Profile.Bindings.Num() > 0 || Profile.Path == SimpleControllerPath)
		{
			// Add the bindings for the controller pose and haptics
			Profile.Bindings.Add(XrActionSuggestedBinding {
				Controllers[EControllerHand::Left].GripAction, GetPath(Instance, "/user/hand/left/input/grip/pose")
				});
			Profile.Bindings.Add(XrActionSuggestedBinding {
				Controllers[EControllerHand::Right].GripAction, GetPath(Instance, "/user/hand/right/input/grip/pose")
				});
			Profile.Bindings.Add(XrActionSuggestedBinding{
				Controllers[EControllerHand::Left].AimAction, GetPath(Instance, "/user/hand/left/input/aim/pose")
				});
			Profile.Bindings.Add(XrActionSuggestedBinding{
				Controllers[EControllerHand::Right].AimAction, GetPath(Instance, "/user/hand/right/input/aim/pose")
				});

			// if (Profile.HasHaptics)
			// {
			// 	Profile.Bindings.Add(XrActionSuggestedBinding{
			// 		Controllers[EControllerHand::Left].VibrationAction, GetPath(Instance, "/user/hand/left/output/haptic")
			// 		});
			// 	Profile.Bindings.Add(XrActionSuggestedBinding{
			// 		Controllers[EControllerHand::Right].VibrationAction, GetPath(Instance, "/user/hand/right/output/haptic")
			// 		});
			// }

			XrInteractionProfileSuggestedBinding InteractionProfile;//自定义手柄绑定类型
			InteractionProfile.type = XR_TYPE_INTERACTION_PROFILE_SUGGESTED_BINDING;
			InteractionProfile.next = nullptr;
			InteractionProfile.interactionProfile = Profile.Path;
			InteractionProfile.countSuggestedBindings = Profile.Bindings.Num();
			InteractionProfile.suggestedBindings = Profile.Bindings.GetData();
			XR_ENSURE(xrSuggestInteractionProfileBindings(Instance, &InteractionProfile));
		}
	}

	Controllers[EControllerHand::Left].AddActionDevices(OpenXRHMD);
	Controllers[EControllerHand::Right].AddActionDevices(OpenXRHMD);

	{
		XrActiveActionSet ActiveSet;
		ActiveSet.actionSet = ActionSet;
		ActiveSet.subactionPath = XR_NULL_PATH;
		ActionSets.Add(ActiveSet);
	}
}

void FYyssXRInputPlugin::FYyssXRInput::DestroyActions()
{
	// Destroying an action set will also destroy all actions in the set
	for (XrActiveActionSet& ActionSet : ActionSets)
	{
		xrDestroyActionSet(ActionSet.actionSet);
	}

	Actions.Reset();
	Controllers.Reset();
	ActionSets.Reset();
}

void FYyssXRInputPlugin::FYyssXRInput::AddKeysToEngine()
{
	EKeys::AddMenuCategoryDisplayInfo("YyssOpenXRButton", LOCTEXT("YyssOpenXRButtonSubCategory", "Yyss OpenXR Button"), TEXT("GraphEditor.PadEvent_16x"));
	
	//left click
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_X_Click, LOCTEXT("Yyss_Left_X_Click", "Yyss (L) X Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Y_Click, LOCTEXT("Yyss_Left_Y_Click", "Yyss (L) Y Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Back_Click, LOCTEXT("Yyss_Left_Back_Click", "Yyss (L) Back Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Home_Click, LOCTEXT("Yyss_Left_Home_Click", "Yyss (L) Home Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Trigger_Click, LOCTEXT("Yyss_Left_Trigger_Click", "Yyss (L) Trigger Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Grip_Click, LOCTEXT("Yyss_Left_Grip_Click", "Yyss (L) Grip Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Rocker_Click, LOCTEXT("Yyss_Left_Rocker_Click", "Yyss (L) Rocker Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey,"YyssOpenXRButton"));
	//right click
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_A_Click, LOCTEXT("Yyss_Right_A_Click", "Yyss (R) A Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_B_Click, LOCTEXT("Yyss_Right_B_Click", "Yyss (R) B Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_Back_Click, LOCTEXT("Yyss_Right_Back_Click", "Yyss (R) Back Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_Home_Click, LOCTEXT("Yyss_Right_Home_Click", "Yyss (R) Home Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_Trigger_Click, LOCTEXT("Yyss_Right_Trigger_Click", "Yyss (R) Trigger Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_Grip_Click, LOCTEXT("Yyss_Right_Grip_Click", "Yyss (R) Grip Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_Rocker_Click, LOCTEXT("Yyss_Right_Rocker_Click", "Yyss (R) Rocker Click"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey,"YyssOpenXRButton"));
	//left touch
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Trigger_Touch, LOCTEXT("Yyss_Left_Trigger_Touch", "Yyss (L) Trigger Touch"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_X_Touch, LOCTEXT("Yyss_Left_X_Touch", "Yyss (L) X Touch"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Y_Touch, LOCTEXT("Yyss_Left_Y_Touch", "Yyss (L) Y Touch"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Rocker_Touch, LOCTEXT("Yyss_Left_Thumbstick_Touch", "Yyss (L) Thumbstick Touch"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	//right touch
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_Trigger_Touch, LOCTEXT("Yyss_Right_Trigger_Touch", "Yyss (R) Trigger Touch"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_A_Touch, LOCTEXT("Yyss_Right_A_Touch", "Yyss (R) A Touch"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_B_Touch, LOCTEXT("Yyss_Right_B_Touch", "Yyss (R) B Touch"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_Rocker_Touch, LOCTEXT("Yyss_Right_Thumbstick_Touch", "Yyss (R) Thumbstick Touch"), FKeyDetails::GamepadKey | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	//left axis
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Trigger_Axis, LOCTEXT("Yyss_Left_Trigger_Axis", "Yyss (L) Trigger Axis"), FKeyDetails::GamepadKey | FKeyDetails::Axis1D | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Rocker_X, LOCTEXT("Yyss_Left_Thumbstick_X", "Yyss (L) Thumbstick X"), FKeyDetails::GamepadKey | FKeyDetails::Axis1D | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Left_Rocker_Y, LOCTEXT("Yyss_Left_Thumbstick_Y", "Yyss (L) Thumbstick Y"), FKeyDetails::GamepadKey | FKeyDetails::Axis1D | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	//right axis
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_Trigger_Axis, LOCTEXT("Yyss_Right_Trigger_Axis", "Yyss (R) Trigger Axis"), FKeyDetails::GamepadKey | FKeyDetails::Axis1D | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_Rocker_X, LOCTEXT("Yyss_Right_Thumbstick_X", "Yyss (R) Thumbstick X"), FKeyDetails::GamepadKey | FKeyDetails::Axis1D | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
	EKeys::AddKey(FKeyDetails(FYyssKey::Yyss_Right_Rocker_Y, LOCTEXT("Yyss_Right_Thumbstick_Y", "Yyss (R) Thumbstick Y"), FKeyDetails::GamepadKey | FKeyDetails::Axis1D | FKeyDetails::NotBlueprintBindableKey, "YyssOpenXRButton"));
}

void FYyssXRInputPlugin::FYyssXRInput::AddYyssAction(XrInstance Instance, XrActionSet ActionSet, XrActionType ActionType, const FName& Name, const FString& ActionPath, bool bIsTrigger)
{
	FOpenXRAction Action(ActionSet, ActionType, Name);
	FInteractionProfile* Profile = Profiles.Find("Yyss");
	Profile->Bindings.Add(XrActionSuggestedBinding{ Action.Handle, GetPath(Instance, ActionPath) });
	Action.bIsTriggerButton = bIsTrigger;
	Actions.Add(Action);
}

void FYyssXRInputPlugin::FYyssXRInput::Tick(float DeltaTime)
{
	if (OpenXRHMD == nullptr)
	{
		// In the editor, when we are not actually running OpenXR, but the IInputDevice exists so it can enumerate its motion sources.
		return;
	}

	XrSession Session = OpenXRHMD->GetSession();
	if (Session != XR_NULL_HANDLE)
	{
		if (!bActionsBound)
		{
			TArray<XrActionSet> BindActionSets;
			for (auto && BindActionSet : ActionSets)
				BindActionSets.Add(BindActionSet.actionSet);

			XrSessionActionSetsAttachInfo SessionActionSetsAttachInfo;
			SessionActionSetsAttachInfo.type = XR_TYPE_SESSION_ACTION_SETS_ATTACH_INFO;
			SessionActionSetsAttachInfo.next = nullptr;
			SessionActionSetsAttachInfo.countActionSets = BindActionSets.Num();
			SessionActionSetsAttachInfo.actionSets = BindActionSets.GetData();
			XR_ENSURE(xrAttachSessionActionSets(Session, &SessionActionSetsAttachInfo));

			bActionsBound = true;
		}
	}
	else if (bActionsBound)
	{
		// If the session shut down, clean up.
		bActionsBound = false;
	}

	if (OpenXRHMD->IsFocused())
	{
		XrActionsSyncInfo SyncInfo;
		SyncInfo.type = XR_TYPE_ACTIONS_SYNC_INFO;
		SyncInfo.next = nullptr;
		SyncInfo.countActiveActionSets = ActionSets.Num();
		SyncInfo.activeActionSets = ActionSets.GetData();
		XR_ENSURE(xrSyncActions(Session, &SyncInfo));
	}
}

namespace OpenXRInputNamespace
{
	FXRTimedInputActionDelegate* GetTimedInputActionDelegate(FName ActionName)
	{
		FXRTimedInputActionDelegate* XRTimedInputActionDelegate = UHeadMountedDisplayFunctionLibrary::OnXRTimedInputActionDelegateMap.Find(ActionName);
		if (XRTimedInputActionDelegate && !XRTimedInputActionDelegate->IsBound())
		{
			XRTimedInputActionDelegate = nullptr;
		}
		return XRTimedInputActionDelegate;
	}
}

void FYyssXRInputPlugin::FYyssXRInput::SendControllerEvents()
{
	if (!bActionsBound)
	{
		return;
	}

	if (OpenXRHMD == nullptr)
	{
		return;
	}

	XrSession Session = OpenXRHMD->GetSession();

	for (FOpenXRAction& Action : Actions)
	{
		XrActionStateGetInfo GetInfo;
		GetInfo.type = XR_TYPE_ACTION_STATE_GET_INFO;
		GetInfo.next = nullptr;
		GetInfo.subactionPath = XR_NULL_PATH;
		GetInfo.action = Action.Handle;

		FName* ActionKey = &Action.Name;
		switch (Action.Type)
		{
		case XR_ACTION_TYPE_BOOLEAN_INPUT:
		{
			XrActionStateBoolean State;
			State.type = XR_TYPE_ACTION_STATE_BOOLEAN;
			State.next = nullptr;
			XrResult Result = xrGetActionStateBoolean(Session, &GetInfo, &State);
			if (XR_SUCCEEDED(Result) && State.changedSinceLastSync)
			{
				if (State.isActive && State.currentState)
				{
					MessageHandler->OnControllerButtonPressed(*ActionKey, 0, /*IsRepeat =*/false);
					UE_LOG(LogInput, Log, TEXT("Input:bool Action %s pressed"), *ActionKey->ToString());
				}
				else
				{
					MessageHandler->OnControllerButtonReleased(*ActionKey, 0, /*IsRepeat =*/false);
					UE_LOG(LogInput, Log, TEXT("Input:bool Action %s released"), *ActionKey->ToString());
				}

				FXRTimedInputActionDelegate* const Delegate = OpenXRInputNamespace::GetTimedInputActionDelegate(Action.Name);
				if (Delegate)
				{
					Delegate->Execute(State.currentState ? 1.0 : 0.0f, ToFTimespan(State.lastChangeTime));
				}
			}
		}
		break;
		case XR_ACTION_TYPE_FLOAT_INPUT:
		{
			XrActionStateFloat State;
			State.type = XR_TYPE_ACTION_STATE_FLOAT;
			State.next = nullptr;
			XrResult Result = xrGetActionStateFloat(Session, &GetInfo, &State);
			if (XR_SUCCEEDED(Result) && State.changedSinceLastSync)
			{
				if (Action.bIsTriggerButton)
				{
					if (State.isActive)
					{
						bool bCurrentTriggerButtonPressed = false;

						if (State.currentState >= 0.5f)
						{
							bCurrentTriggerButtonPressed = true;
						}

						if (bCurrentTriggerButtonPressed != Action.bTriggerButtonPressed)
						{
							Action.bTriggerButtonPressed = bCurrentTriggerButtonPressed;
							if (Action.bTriggerButtonPressed)
							{
								MessageHandler->OnControllerButtonPressed(*ActionKey, 0, /*IsRepeat =*/false);
								UE_LOG(LogInput, Log, TEXT("Input:float Action %s pressed"), *ActionKey->ToString());
							}
							else
							{
								MessageHandler->OnControllerButtonReleased(*ActionKey, 0, /*IsRepeat =*/false);
								UE_LOG(LogInput, Log, TEXT("Input:float Action %s released"), *ActionKey->ToString());
							}
						}
					}
					else
					{
						MessageHandler->OnControllerButtonReleased(*ActionKey, 0, /*IsRepeat =*/false);
						UE_LOG(LogInput, Log, TEXT("Input:float Action %s released"), *ActionKey->ToString());
					}
				}
				else
				{
					if (State.isActive)
					{
						MessageHandler->OnControllerAnalog(*ActionKey, 0, State.currentState);
						UE_LOG(LogInput, Log, TEXT("Input:float Action %s Analog %f"), *ActionKey->ToString(), State.currentState);
					}
					else
					{
						MessageHandler->OnControllerAnalog(*ActionKey, 0, 0.0f);
						UE_LOG(LogInput, Log, TEXT("Input:float Action %s Analog %f"), *ActionKey->ToString(), 0.0f);
					}
				}
			}
		}
		break;
		default:
			// Other action types are currently unsupported.
			break;
		}
	}
}

void FYyssXRInputPlugin::FYyssXRInput::SetMessageHandler(const TSharedRef< FGenericApplicationMessageHandler >& InMessageHandler)
{
	MessageHandler = InMessageHandler;
#if WITH_EDITOR
	FEditorDelegates::OnActionAxisMappingsChanged.AddSP(this, &FYyssXRInputPlugin::FYyssXRInput::BuildActions);
#endif
}

bool FYyssXRInputPlugin::FYyssXRInput::Exec(UWorld* InWorld, const TCHAR* Cmd, FOutputDevice& Ar)
{
	return false;
}

void FYyssXRInputPlugin::FYyssXRInput::SetChannelValue(int32 ControllerId, FForceFeedbackChannelType ChannelType, float Value)
{
	// Large channel type maps to amplitude. We are interested in amplitude.
	if ((ChannelType == FForceFeedbackChannelType::LEFT_LARGE) ||
		(ChannelType == FForceFeedbackChannelType::RIGHT_LARGE))
	{
		FHapticFeedbackValues Values(XR_FREQUENCY_UNSPECIFIED, Value);
		SetHapticFeedbackValues(ControllerId, ChannelType == FForceFeedbackChannelType::LEFT_LARGE ? (int32)EControllerHand::Left : (int32)EControllerHand::Right, Values);
	}
}

void FYyssXRInputPlugin::FYyssXRInput::SetChannelValues(int32 ControllerId, const FForceFeedbackValues &values)
{
	FHapticFeedbackValues leftHaptics = FHapticFeedbackValues(
		values.LeftSmall,		// frequency
		values.LeftLarge);		// amplitude
	FHapticFeedbackValues rightHaptics = FHapticFeedbackValues(
		values.RightSmall,		// frequency
		values.RightLarge);		// amplitude

	SetHapticFeedbackValues(
		ControllerId,
		(int32)EControllerHand::Left,
		leftHaptics);

	SetHapticFeedbackValues(
		ControllerId,
		(int32)EControllerHand::Right,
		rightHaptics);
}

FName FYyssXRInputPlugin::FYyssXRInput::GetMotionControllerDeviceTypeName() const
{
	return FName(TEXT("OpenXR"));
}


bool FYyssXRInputPlugin::FYyssXRInput::GetControllerOrientationAndPosition(const int32 ControllerIndex, const FName MotionSource, FRotator& OutOrientation, FVector& OutPosition, float WorldToMetersScale) const
{
	if (OpenXRHMD == nullptr)
	{
		return false;
	}

	if (ControllerIndex == 0 && IsOpenXRInputSupportedMotionSource(MotionSource))
	{
		if (MotionSource == OpenXRSourceNames::AnyHand)
		{
			return GetControllerOrientationAndPosition(ControllerIndex, OpenXRSourceNames::LeftGrip, OutOrientation, OutPosition, WorldToMetersScale)
				|| GetControllerOrientationAndPosition(ControllerIndex, OpenXRSourceNames::RightGrip, OutOrientation, OutPosition, WorldToMetersScale);
		}

		if (MotionSource == OpenXRSourceNames::Left)
		{
			return GetControllerOrientationAndPosition(ControllerIndex, OpenXRSourceNames::LeftGrip, OutOrientation, OutPosition, WorldToMetersScale);
		}

		if (MotionSource == OpenXRSourceNames::Right)
		{
			return GetControllerOrientationAndPosition(ControllerIndex, OpenXRSourceNames::RightGrip, OutOrientation, OutPosition, WorldToMetersScale);
		}

		XrSession Session = OpenXRHMD->GetSession();

		if (Session == XR_NULL_HANDLE)
		{
			return false;
		}

		XrActionStateGetInfo GetInfo;
		GetInfo.type = XR_TYPE_ACTION_STATE_GET_INFO;
		GetInfo.next = nullptr;
		GetInfo.subactionPath = XR_NULL_PATH;
		GetInfo.action = GetActionForMotionSource(MotionSource);

		XrActionStatePose State;
		State.type = XR_TYPE_ACTION_STATE_POSE;
		State.next = nullptr;
		XrResult Result = xrGetActionStatePose(Session, &GetInfo, &State);
		if (Result >= XR_SUCCESS && State.isActive)
		{
			FQuat Orientation;
			OpenXRHMD->GetCurrentPose(GetDeviceIDForMotionSource(MotionSource), Orientation, OutPosition);
			OutOrientation = FRotator(Orientation);
			return true;
		}
	}

	return false;
}

bool FYyssXRInputPlugin::FYyssXRInput::GetControllerOrientationAndPositionForTime(const int32 ControllerIndex, const FName MotionSource, FTimespan Time, bool& OutTimeWasUsed, FRotator& OutOrientation, FVector& OutPosition, bool& OutbProvidedLinearVelocity, FVector& OutLinearVelocity, bool& OutbProvidedAngularVelocity, FVector& OutAngularVelocityRadPerSec, float WorldToMetersScale) const
{
	OutTimeWasUsed = true;

	if (OpenXRHMD == nullptr)
	{
		return false;
	}

	if (ControllerIndex == 0 && IsOpenXRInputSupportedMotionSource(MotionSource))
	{
		if (MotionSource == OpenXRSourceNames::AnyHand)
		{
			return GetControllerOrientationAndPositionForTime(ControllerIndex, OpenXRSourceNames::LeftGrip, Time, OutTimeWasUsed, OutOrientation, OutPosition, OutbProvidedLinearVelocity, OutLinearVelocity, OutbProvidedAngularVelocity, OutAngularVelocityRadPerSec, WorldToMetersScale)
				|| GetControllerOrientationAndPositionForTime(ControllerIndex, OpenXRSourceNames::RightGrip, Time, OutTimeWasUsed, OutOrientation, OutPosition, OutbProvidedLinearVelocity, OutLinearVelocity, OutbProvidedAngularVelocity, OutAngularVelocityRadPerSec, WorldToMetersScale);
		}

		if (MotionSource == OpenXRSourceNames::Left)
		{
			return GetControllerOrientationAndPositionForTime(ControllerIndex, OpenXRSourceNames::LeftGrip, Time, OutTimeWasUsed, OutOrientation, OutPosition, OutbProvidedLinearVelocity, OutLinearVelocity, OutbProvidedAngularVelocity, OutAngularVelocityRadPerSec, WorldToMetersScale);
		}

		if (MotionSource == OpenXRSourceNames::Right)
		{
			return GetControllerOrientationAndPositionForTime(ControllerIndex, OpenXRSourceNames::RightGrip, Time, OutTimeWasUsed, OutOrientation, OutPosition, OutbProvidedLinearVelocity, OutLinearVelocity, OutbProvidedAngularVelocity, OutAngularVelocityRadPerSec, WorldToMetersScale);
		}
	}

	XrSession Session = OpenXRHMD->GetSession();

	if (Session == XR_NULL_HANDLE)
	{
		return false;
	}

	XrActionStateGetInfo GetInfo;
	GetInfo.type = XR_TYPE_ACTION_STATE_GET_INFO;
	GetInfo.next = nullptr;
	GetInfo.subactionPath = XR_NULL_PATH;
	GetInfo.action = GetActionForMotionSource(MotionSource);

	XrActionStatePose State;
	State.type = XR_TYPE_ACTION_STATE_POSE;
	State.next = nullptr;
	XrResult Result = xrGetActionStatePose(Session, &GetInfo, &State);
	if (Result >= XR_SUCCESS && State.isActive)
	{
		FQuat Orientation;
		OpenXRHMD->GetPoseForTime(GetDeviceIDForMotionSource(MotionSource), Time, Orientation, OutPosition, OutbProvidedLinearVelocity, OutLinearVelocity, OutbProvidedAngularVelocity, OutAngularVelocityRadPerSec);
		OutOrientation = FRotator(Orientation);
		return true;
	}

	return false;
}

ETrackingStatus FYyssXRInputPlugin::FYyssXRInput::GetControllerTrackingStatus(const int32 ControllerIndex, const FName MotionSource) const
{
	if (OpenXRHMD == nullptr)
	{
		return ETrackingStatus::NotTracked;
	}

	if (ControllerIndex == 0 && IsOpenXRInputSupportedMotionSource(MotionSource))
	{
		if (MotionSource == OpenXRSourceNames::AnyHand)
		{
			if (GetControllerTrackingStatus(ControllerIndex, OpenXRSourceNames::LeftGrip) == ETrackingStatus::Tracked)
			{
				return ETrackingStatus::Tracked;
			}
			else
			{
				return GetControllerTrackingStatus(ControllerIndex, OpenXRSourceNames::RightGrip);
			}
		}

		if (MotionSource == OpenXRSourceNames::Left)
		{
			return GetControllerTrackingStatus(ControllerIndex, OpenXRSourceNames::LeftGrip);
		}

		if (MotionSource == OpenXRSourceNames::Right)
		{
			return GetControllerTrackingStatus(ControllerIndex, OpenXRSourceNames::RightGrip);
		}

		XrSession Session = OpenXRHMD->GetSession();

		if (Session == XR_NULL_HANDLE)
		{
			return ETrackingStatus::NotTracked;
		}

		XrActionStateGetInfo GetInfo;
		GetInfo.type = XR_TYPE_ACTION_STATE_GET_INFO;
		GetInfo.next = nullptr;
		GetInfo.subactionPath = XR_NULL_PATH;
		GetInfo.action = GetActionForMotionSource(MotionSource);

		XrActionStatePose State;
		State.type = XR_TYPE_ACTION_STATE_POSE;
		State.next = nullptr;
		XrResult Result = xrGetActionStatePose(Session, &GetInfo, &State);
		if (XR_SUCCEEDED(Result) && State.isActive)
		{
			FQuat Orientation;
			bool bIsTracked = OpenXRHMD->GetIsTracked(GetDeviceIDForMotionSource(MotionSource));
			return bIsTracked ? ETrackingStatus::Tracked : ETrackingStatus::NotTracked;
		}
	}

	return ETrackingStatus::NotTracked;
}

void FYyssXRInputPlugin::FYyssXRInput::EnumerateSources(TArray<FMotionControllerSource>& SourcesOut) const
{
	check(IsInGameThread());

	SourcesOut.Add(OpenXRSourceNames::AnyHand);
	SourcesOut.Add(OpenXRSourceNames::Left);
	SourcesOut.Add(OpenXRSourceNames::Right);
	SourcesOut.Add(OpenXRSourceNames::LeftGrip);
	SourcesOut.Add(OpenXRSourceNames::RightGrip);
	SourcesOut.Add(OpenXRSourceNames::LeftAim);
	SourcesOut.Add(OpenXRSourceNames::RightAim);
}

void FYyssXRInputPlugin::FYyssXRInput::SetHapticFeedbackValues(int32 ControllerId, int32 Hand, const FHapticFeedbackValues& Values)
{
	if (OpenXRHMD == nullptr)
	{
		return;
	}

	XrSession Session = OpenXRHMD->GetSession();

	if (Session == XR_NULL_HANDLE)
	{
		return;
	}

	XrHapticVibration HapticValue;
	HapticValue.type = XR_TYPE_HAPTIC_VIBRATION;
	HapticValue.next = nullptr;
	HapticValue.duration = MaxFeedbackDuration;
	HapticValue.frequency = Values.Frequency;
	HapticValue.amplitude = Values.Amplitude;

	if (ControllerId == 0)
	{
		if ((Hand == (int32)EControllerHand::Left || Hand == (int32)EControllerHand::AnyHand) && Controllers.Contains(EControllerHand::Left))
		{
			XrHapticActionInfo HapticActionInfo;
			HapticActionInfo.type = XR_TYPE_HAPTIC_ACTION_INFO;
			HapticActionInfo.next = nullptr;
			HapticActionInfo.subactionPath = XR_NULL_PATH;
			HapticActionInfo.action = Controllers[EControllerHand::Left].VibrationAction;
			if (Values.Amplitude <= 0.0f || Values.Frequency < XR_FREQUENCY_UNSPECIFIED)
			{
				XR_ENSURE(xrStopHapticFeedback(Session, &HapticActionInfo));
			}
			else
			{
				XR_ENSURE(xrApplyHapticFeedback(Session, &HapticActionInfo, (const XrHapticBaseHeader*)&HapticValue));
			}
		}
		if ((Hand == (int32)EControllerHand::Right || Hand == (int32)EControllerHand::AnyHand) && Controllers.Contains(EControllerHand::Right))
		{
			XrHapticActionInfo HapticActionInfo;
			HapticActionInfo.type = XR_TYPE_HAPTIC_ACTION_INFO;
			HapticActionInfo.next = nullptr;
			HapticActionInfo.subactionPath = XR_NULL_PATH;
			HapticActionInfo.action = Controllers[EControllerHand::Right].VibrationAction;
			if (Values.Amplitude <= 0.0f || Values.Frequency < XR_FREQUENCY_UNSPECIFIED)
			{
				XR_ENSURE(xrStopHapticFeedback(Session, &HapticActionInfo));
			}
			else
			{
				XR_ENSURE(xrApplyHapticFeedback(Session, &HapticActionInfo, (const XrHapticBaseHeader*)&HapticValue));
			}
		}
	}
}

void FYyssXRInputPlugin::FYyssXRInput::GetHapticFrequencyRange(float& MinFrequency, float& MaxFrequency) const
{
	MinFrequency = XR_FREQUENCY_UNSPECIFIED;
	MaxFrequency = XR_FREQUENCY_UNSPECIFIED;
}

float FYyssXRInputPlugin::FYyssXRInput::GetHapticAmplitudeScale() const
{
	return 1.0f;
}

#undef LOCTEXT_NAMESPACE // "OpenXRInputPlugin"
