// Copyright 2020-2023 Mesiontech Technology Co., Ltd. All Rights RHome.

#pragma once
#include "YyssXRCore.h"
#include "GenericPlatform/IInputInterface.h"
#include "XRMotionControllerBase.h"
#include "IYyssXRInputPlugin.h"
#include "IInputDevice.h"
#include "IHapticDevice.h"

class FYyssXRHMD;
struct FInputActionKeyMapping;
struct FInputAxisKeyMapping;

class FYyssXRInputPlugin : public IYyssXRInputPlugin
{
public:
	struct FOpenXRAction
	{
		XrActionSet		Set;
		XrActionType	Type;
		FName			Name;
		XrAction		Handle;
		bool bIsTriggerButton;
		bool bTriggerButtonPressed;

		FOpenXRAction(XrActionSet InActionSet, XrActionType InActionType, const FName& InName);
	};

	struct FOpenXRController
	{
		XrActionSet		ActionSet;
		XrPath			UserPath;
		XrAction		GripAction;
		XrAction		AimAction;
		XrAction		VibrationAction;
		int32			GripDeviceId;
		int32			AimDeviceId;

		FOpenXRController(XrActionSet InActionSet, XrPath InUserPath, const char* InName);

		void AddActionDevices(FYyssXRHMD* HMD);
	};

	struct FInteractionProfile
	{
	public:
		bool HasHaptics;
		XrPath Path;
		TArray<XrActionSuggestedBinding> Bindings;

		FInteractionProfile(XrPath InProfile, bool InHasHaptics);
	};

	class FYyssXRInput : public IInputDevice, public FXRMotionControllerBase, public IHapticDevice, public TSharedFromThis<FYyssXRInput>
	{
	public:
		FYyssXRInput(FYyssXRHMD* HMD);
		virtual ~FYyssXRInput();

		// IInputDevice overrides
		virtual void Tick(float DeltaTime) override;
		virtual void SendControllerEvents() override;
		virtual void SetMessageHandler(const TSharedRef< FGenericApplicationMessageHandler >& InMessageHandler) override;
		virtual bool Exec(UWorld* InWorld, const TCHAR* Cmd, FOutputDevice& Ar) override;
		virtual void SetChannelValue(int32 ControllerId, FForceFeedbackChannelType ChannelType, float Value) override;
		virtual void SetChannelValues(int32 ControllerId, const FForceFeedbackValues &values) override;

		// IMotionController overrides
		virtual FName GetMotionControllerDeviceTypeName() const override;
		virtual bool GetControllerOrientationAndPosition(const int32 ControllerIndex, const FName MotionSource, FRotator& OutOrientation, FVector& OutPosition, float WorldToMetersScale) const override;
		virtual bool GetControllerOrientationAndPositionForTime(const int32 ControllerIndex, const FName MotionSource, FTimespan Time, bool& OutTimeWasUsed, FRotator& OutOrientation, FVector& OutPosition, bool& OutbProvidedLinearVelocity, FVector& OutLinearVelocity, bool& OutbProvidedAngularVelocity, FVector& OutAngularVelocityRadPerSec, float WorldToMetersScale) const override;
		virtual ETrackingStatus GetControllerTrackingStatus(const int32 ControllerIndex, const FName MotionSource) const override;
		virtual bool GetControllerOrientationAndPosition(const int32 ControllerIndex, const EControllerHand DeviceHand, FRotator& OutOrientation, FVector& OutPosition, float WorldToMetersScale) const override { check(false); return false; }
		virtual ETrackingStatus GetControllerTrackingStatus(const int32 ControllerIndex, const EControllerHand DeviceHand) const override { check(false); return ETrackingStatus::NotTracked; }
		virtual void EnumerateSources(TArray<FMotionControllerSource>& SourcesOut) const override;

		// IHapticDevice overrides
		IHapticDevice* GetHapticDevice() override { return (IHapticDevice*)this; }
		virtual void SetHapticFeedbackValues(int32 ControllerId, int32 Hand, const FHapticFeedbackValues& Values) override;

		virtual void GetHapticFrequencyRange(float& MinFrequency, float& MaxFrequency) const override;
		virtual float GetHapticAmplitudeScale() const override;

	private:
		static const XrDuration MaxFeedbackDuration = 2500000000; // 2.5 seconds

		FYyssXRHMD* OpenXRHMD;

		TArray<XrActiveActionSet> ActionSets;
		TMap<FString, FInteractionProfile> Profiles;
		TArray<FOpenXRAction> Actions;
		TMap<EControllerHand, FOpenXRController> Controllers;
		TMap<FName, EControllerHand> MotionSourceToControllerHandMap;
		XrAction GetActionForMotionSource(FName MotionSource) const;
		int32 GetDeviceIDForMotionSource(FName MotionSource) const;
		XrPath GetUserPathForMotionSource(FName MotionSource) const;
		bool IsOpenXRInputSupportedMotionSource(const FName MotionSource) const;
		bool bActionsBound;

		void BuildActions();
		void DestroyActions();
		void AddKeysToEngine();

		void AddYyssAction(XrInstance Instance, XrActionSet ActionSet, XrActionType ActionType, const FName& Name, const FString& ActionPath, bool bIsTrigger = false);

		/** handler to send all messages to */
		TSharedRef<FGenericApplicationMessageHandler> MessageHandler;
	};

	FYyssXRInputPlugin();
	virtual ~FYyssXRInputPlugin();

	virtual void StartupModule() override;
	virtual TSharedPtr< class IInputDevice > CreateInputDevice(const TSharedRef< FGenericApplicationMessageHandler >& InMessageHandler) override;

private:
	FYyssXRHMD* GetOpenXRHMD() const;

private:
	TSharedPtr<FYyssXRInput> InputDevice;
};
